#pragma rtGlobals=1		// Use modern global access method.
#pragma version=3.1

// *********** Kaiser Maximally Flat **********

Function KaiserMaxFlatReset()
	KaiserMaxFlatSetupDesign()
	SetAxis/A/Z bottom			// in case user set a manual X axis range
	Execute "kmf_compute(\"\")"
End

Function ButtonKaiserMaxFlat(ctrlName) : ButtonControl
	String ctrlName

	DoWindow/F WMKaiserMaxFlatDesign
	if( V_Flag==0 )
		KaiserMaxFlatSetupDesign()
	endif
End

Function KaiserMaxFlatSetupDesign()

	SetApplyButtonTitle(0)

	NVAR fs= root:Packages:WM_IFDL:fs
	String dfSav= Set_IFDL_DataFolder()

	Variable beta=NumVarOrDefault("root:Packages:WM_IFDL:kmf_beta",0.2)	// transition frequency
	Variable gamma=NumVarOrDefault("root:Packages:WM_IFDL:kmf_gamma",0.1) // transition width

	// Create the globals
	Variable/G kmf_beta= beta
	Variable/G kmf_gamma= gamma
	Variable/G kmf_n

	// scale the saved normalized frequencies for the SetVariable controls
	Variable/G kmf_beta_fs= beta * fs
	Variable/G kmf_gamma_fs= gamma * fs
	
	SetDataFolder dfSav

	KaiserMaxFlatUpdate()
	
	CreateKaiserMaxFlatDesign()
	KaiserMaxFlatUpdate()			// set the control limits, too.
End

Function KaiserMaxFlatUpdate()
	String dfSav= Set_IFDL_DataFolder()
	NVAR fs= fs
	Variable betaFs=NumVarOrDefault("kmf_beta_fs",0.2*fs)
	Variable gammaFs=NumVarOrDefault("kmf_gamma_fs",0.1*fs)
	
	// Validate these values that are controlled by SetVariable controls
	// The allowable width is a function of the center frequency
	// and of the maximum number of terms
	Variable beta= betaFs/fs
	Variable gamma= gammaFs/fs
		
	// back-compute minimum transition width (gamma)
	// from maximum number of terms:
	// 		Variable maxTerms=ceil(5/16/minGamma/minGamma)+1
	//		maxTerms -1 = ceil(5/(16*minGamma*minGamma))
	//		16*minGamma*minGamma = 5/maxTerms
	Variable maxTerms= 1000-2	// extra debit because of ceil()
	Variable minGamma= sqrt(5/(16*maxTerms))

	Variable margin = minGamma /4		 // don't try to set edge of transition band at 0 or fs/2, leave this much margin

	// maximum width depends on center frequency
	Variable maxGamma = 2*min(beta-margin,(0.5-margin)-beta)

	// limit the transition width (gamma) control
	gamma= limit(gamma,minGamma,maxGamma)
	gamma= limit(gamma,0,0.5-margin*2)
	gammaFs= gamma * fs

	// limit the center frequency (beta) control so that beta-gamma/2 > 0 and beta+gamma/2 < fs/2
	Variable minBeta = margin + gamma / 2			// less than this would push the transition region below 0 +margin Hz
	Variable maxBeta = (0.5 -margin)- gamma / 2	// more than this would push the transition region above fs/2 -margin

	beta= limit(beta,minBeta,maxBeta)
	beta= limit(beta,0,0.5)
	betaFs= beta * fs

	// Save limited/verified values
	Variable/G kmf_beta_fs= betaFs
	Variable/G kmf_gamma_fs= gammaFs

	// Save the normalized frequencies in case the user changes sampling frequency
	Variable/G kmf_beta= beta
	Variable/G kmf_gamma= gamma

	// Passband and stop band response pair, linear
	Variable b1endFs= max(0,betaFs-gammaFs/2)
	Variable b2startFs= min(betaFs+gammaFs/2,fs/2)
	Make/O kmf_responseX = {0,b1endFs,NaN,b2startFs,fs*0.5}
	Make/O kmf_response  = {1,1,NaN,1e-6,1e-6}	// updated from coefsMag

	Wave filterResult= root:Packages:WM_IFDL:coefsMag
	DoWindow WMKaiserMaxFlatDesign
	if( V_Flag == 1 )
		CheckDisplayed/W=WMKaiserMaxFlatDesign filterResult
		if( V_Flag == 1 )
			Variable minResp= filterResult[numpnts(filterResult)-2]
			kmf_response[3,4] = {minResp,minResp}
		endif
		ControlInfo/W=WMKaiserMaxFlatDesign dbCheck
		if( V_Value ) 	// want DB
			kmf_response= 20*log(kmf_response)
		endif
	endif

	// transition region waves, linear
	Make/O     kmf_transitionX = {kmf_responseX[1],kmf_responseX[3]}
	Make/O/N=2 kmf_transitionPlus =  max(kmf_response[0],kmf_response[3])
	Make/O/N=2 kmf_transitionMinus = min(kmf_response[0],kmf_response[3])

	DoWindow WMKaiserMaxFlatDesign
	if( V_Flag == 1 )
		Variable df=  NiceNumber(fs/200)
		SetVariable kmf_beta,limits={minBeta*fs,maxBeta*fs,df},win=WMKaiserMaxFlatDesign
		df=  NiceNumber(maxGamma*fs/20)
		SetVariable kmf_gamma,limits={minGamma*fs,maxGamma*fs,df},win=WMKaiserMaxFlatDesign
		ControlUpdate/A/W=WMKaiserMaxFlatDesign
	endif
	SetDataFolder dfSav
End

Function CreateKaiserMaxFlatDesign()

	if( DesignGraph("WMKaiserMaxFlatDesign","Kaiser Maximally Flat Design") )
		return 1	// already existed, don't reset the graph settings.
	endif

	NVAR fs= root:Packages:WM_IFDL:fs
	Variable df= NiceNumber(fs/200)	// 1,2,5 increment for SetVariable controls

	String dfSav= Set_IFDL_DataFolder()
	AppendToGraph/L=responseLeft  kmf_transitionPlus,kmf_transitionMinus vs kmf_transitionX
	AppendToGraph/L=responseLeft kmf_response vs kmf_responseX
	SetDataFolder dfSav

	ModifyGraph mode(kmf_transitionPlus)=7
	ModifyGraph lSize(kmf_transitionPlus)=0,lSize(kmf_transitionMinus)=0
	ModifyGraph rgb(kmf_transitionPlus)=(56797,56797,56797)
	ModifyGraph hbFill(kmf_transitionPlus)=2
	ModifyGraph toMode(kmf_transitionPlus)=1
	ModifyGraph lblPos(responseLeft)=56
	ModifyGraph freePos(responseLeft)={0,bottom}
	ModifyGraph margin(left)=67
	ModifyGraph minor(bottom)=1
	SetAxis/A/N=1 responseLeft
	Label responseLeft "response"
	ControlBar 61
	SetVariable kmf_beta,pos={14,3},size={235,17},proc=KaiserMF,title="Transition Frequency"
	SetVariable kmf_beta,limits={0,fs/2,df},value= root:Packages:WM_IFDL:kmf_beta_fs
	SetVariable kmf_gamma,pos={40,22},size={209,17},proc=KaiserMF,title="Transition Width"
	SetVariable kmf_gamma,limits={0,fs/2,df},value= root:Packages:WM_IFDL:kmf_gamma_fs
	CheckBox dbCheck,pos={130,40},size={118,20},proc=kmfDbCheck,title="dB Response",value=0
	Button kmfCompute,pos={328,12},size={109,19},proc=kmf_compute,title="Compute Filter"

	Textbox/N=kmfLegend/X=0.74/Y=1.60

	kmfLegend()
	SetWindow WMKaiserMaxFlatDesign, hook=designHook
	return 0
End


Function kmfDbCheck(ctrlName,checked) : CheckBoxControl
	String ctrlName
	Variable checked
 	
 	dBCheck(checked)
	KaiserMaxFlatUpdate()
	kmfLegend()
End

Function kmfLegend()
	DesignLegend("WMKaiserMaxFlatDesign","kmf","kmfLegend")
End

Function KaiserMF(ctrlName,varNum,varStr,varName) : SetVariableControl
	String ctrlName
	Variable varNum
	String varStr
	String varName
	
	KaiserMaxFlatUpdate()
End

Proc kmf_compute(ctrlName) : ButtonControl
	String ctrlName

	Silent 1;PauseUpdate	// Kaiser Maximally Flat...
	String dfSav= Set_IFDL_DataFolder()
	Variable beta=NumVarOrDefault("root:Packages:WM_IFDL:kmf_beta",0.2)	// transition frequency
	Variable gamma=NumVarOrDefault("root:Packages:WM_IFDL:kmf_gamma",0.1) // transition width
	SetDataFolder dfSav
	Variable fs = root:Packages:WM_IFDL:fs
	
	if( ChkFreq(beta,gamma,0,0) )
		abort
	endif
	Variable maxGamma=min(2*beta,1-2*beta)

	if(gamma>maxGamma)
		abort "transition width too big for transition frequency; max width is "+num2str(maxGamma*fs)
	endif

	Variable maxt=ceil(5/16/gamma/gamma)+1
	if(maxt>1000)
		abort "transition width too small:  more than 2000 coefficients would be required!"
	endif
	Set_IFDL_DataFolder()
	String/G proposedFilterName="kaiserMaxFlat"
	String/G designTypeName="Kaiser Maximally Flat; low pass"
	Variable/G designFlags=0x3
	Make/O bandInfo={1,0,(beta-gamma/2)*fs,0,(beta+gamma/2)*fs,0.5*fs}
	SetDataFolder dfSav

	Make/O/N=(maxt+1) coefs
	IFDLKaiserMaxFlat(beta,gamma,"coefs")	// changes number of points in coefs
	StdCoefsTreatmentNoShowResults(0xf,1)

	DoWindow/F WMKaiserMaxFlatDesign

	Variable ncoefs= numpnts(coefs)
	root:Packages:WM_IFDL:kmf_n = ncoefs
	ValDisplay kmfTerms,pos={315,37},size={149,17},title="Computed Terms:",frame=0
	ValDisplay kmfTerms,limits={0,0,0},barmisc={0,1000}, frame=(IsWindowsPlatform())
	ValDisplay kmfTerms,value= #"root:Packages:WM_IFDL:kmf_n"

	CheckDisplayed/W=WMKaiserMaxFlatDesign root:Packages:WM_IFDL:coefsMag,root:Packages:WM_IFDL:coefsDbMag
	if( V_Flag == 0 )
		ControlInfo/W=WMKaiserMaxFlatDesign dbCheck
		Variable wantDB= V_value
		Append/L=responseLeft $responseName(wantDB,1)
		String traceName= responseName(wantDB,0)
		ModifyGraph rgb($traceName)=(0,0,65535)
		ReorderTraces kmf_response, {$traceName}	// put actual response under desired response
	endif
	AppendPassDetails("detailsLeft",root:Packages:WM_IFDL:magPassDetails)
	AutoApplyFilter()
	KaiserMaxFlatUpdate()
	kmfLegend()
end
